/* Main code for Preference editor */

/* $Id: PrefEd.c,v 1.25 2000/10/07 18:10:24 bodo Exp $ */
/*
 * Copyright (C) 1998 Bodo Bellut
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:  Bodo Bellut (bodo@bellut.net)
 */

#include <Pilot.h>
#include "callback.h"

#include "PrefEdRsc.h"

#undef DISPLAY_ID_TOO

typedef struct { ULong   creator;
#ifdef DISPLAY_ID_TOO
                 Int     id;
#endif
                 CharPtr name;
               } listItemType;
               
typedef enum { SAVED = 0,
               UNSAVED
             } prefsType;

/* globals */
#define MEMOMAX 4093
#ifdef DISPLAY_ID_TOO
#  define MEMOLINE (32 + 5 + 3 + 6 + 7)
#else
#  define MEMOLINE (32 + 5 + 3 + 6)
#endif
#define MAXDB 400
listItemType items[2][MAXDB];
int nitems[2];
int currentItem[2] = { -1, -1};
int currentTopItem[2] = { -1, -1};

prefsType ptype = SAVED;

/* globals for view function */
unsigned char *ptr = NULL;
VoidHand hand = NULL;
ULong offset;
#define LINEWIDTH 8
#define LINEHEIGHT 10

/* helper functions */

static VoidPtr GetObjectPtr(Word objectID)
{
    FormPtr frm = FrmGetActiveForm();
    
    return FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, objectID));
}

static Word LstGetTopItem(const ListPtr pList)
{
    return (pList->topItem);
}

/* main code */

static void DrawPrefList(UInt itemNum, RectanglePtr bounds, CharPtr *unusedP)
{
    CharPtr name = NULL;
    Char    creator[5];
#ifdef DISPLAY_ID_TOO
    Char    id[6];
#endif

    CALLBACK_PROLOGUE
    
    name = items[ptype][itemNum].name;
#ifdef DISPLAY_ID_TOO
    StrIToA(id, (UInt) (items[ptype][itemNum].id));
#endif
    MemMove(creator, &items[ptype][itemNum].creator, sizeof(items[ptype][itemNum].creator));

    creator[4] = '\0';
    ErrFatalDisplayIf(!name, "Oops");
    
    WinDrawChars(creator, StrLen(creator), bounds->topLeft.x, bounds->topLeft.y);
#ifdef DISPLAY_ID_TOO
    WinDrawChars(id, StrLen(id)    , bounds->topLeft.x + 30, bounds->topLeft.y);
    WinDrawChars(name, StrLen(name), bounds->topLeft.x + 60, bounds->topLeft.y);
#else
    WinDrawChars(name, StrLen(name), bounds->topLeft.x + 30, bounds->topLeft.y);
#endif
    
    CALLBACK_EPILOGUE
}

static void InitPrefList2(prefsType type, UInt cardNo, LocalID dbID);

static void InitPrefList(void)
{
    Err			err;
    UInt		cardNo;
    LocalID		dbID;
    DmSearchStateType	state;
    Boolean		newSearch = 1;
    
    err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'sprf', 'psys', false, &cardNo, &dbID);

    ErrFatalDisplayIf(err == dmErrCantFind, "No Saved Preferences");
    
    InitPrefList2(SAVED, cardNo, dbID);

    err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'pref', 'psys', false, &cardNo, &dbID);
    
    ErrFatalDisplayIf(err == dmErrCantFind, "No Unsaved Preferences");
    
    InitPrefList2(UNSAVED, cardNo, dbID);
}

static void InitPrefList2(prefsType type, UInt cardNo, LocalID dbID)
{
    Err		err;
    DmOpenRef	ref;
    DmSearchStateType	state;
    Boolean	newSearch = 1;
    ULong	numRecords;
    ULong	resType;
#ifdef DISPLAY_ID_TOO
    Int		resID;
#endif
    int		j;
    Char	tmpbuf[50];
    

    ref = DmOpenDatabase(cardNo, dbID, dmModeReadOnly);
    
    if (ref == 0) {
	err = DmGetLastErr();
	StrCopy(tmpbuf, "DmOpenDatabase: condition ");
	StrIToA((tmpbuf + StrLen(tmpbuf)), err);
	ErrFatalDisplay(tmpbuf);
    }
    
    err = DmDatabaseSize(cardNo, dbID, &numRecords, NULL, NULL);
    
    ErrFatalDisplayIf(err != 0, "DmDatabaseSize");

    for (j=0; j < nitems[type]; j++) {
	MemPtrFree(items[type][j].name);
    }
    nitems[type] = 0;

    for (nitems[type]=0; nitems[type] < numRecords && nitems[type] < MAXDB-1; nitems[type]++) {
    
	char name[50];
    
	items[type][nitems[type]].name = MemPtrNew(40);
	
	ErrFatalDisplayIf(items[type][nitems[type]].name == NULL, "MemPtrNew");
	
	MemSet(items[type][nitems[type]].name, 40, 0);
#ifdef DISPLAY_ID_TOO
	err = DmResourceInfo(ref, nitems[type], &resType, &resID, NULL);
#else
	err = DmResourceInfo(ref, nitems[type], &resType, NULL, NULL);
#endif
	ErrFatalDisplayIf(err, "Couldn't open record");
	items[type][nitems[type]].creator = resType;
#ifdef DISPLAY_ID_TOO
	items[type][nitems[type]].id      = resID;
#endif
	if (resType == 'psys') {
	    StrCopy(items[type][nitems[type]].name, "PalmOS");
	} else {
	    err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'appl', resType, false, &cardNo, &dbID);
	    if (err == dmErrCantFind) {
		err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'HACK', resType, false, &cardNo, &dbID);
	    }
	    if (err == dmErrCantFind) {
		err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'panl', resType, false, &cardNo, &dbID);
	    }
	    if (err == dmErrCantFind) {
		err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'exgl', resType, false, &cardNo, &dbID);
	    }
	    if (err == dmErrCantFind) {
		err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'DAcc', resType, false, &cardNo, &dbID);
	    }
	    if (err == dmErrCantFind) {
		StrCopy(items[type][nitems[type]].name, "-none-");
	    } else {
		err = DmDatabaseInfo(cardNo, dbID, name, NULL, NULL, NULL, NULL, NULL,
	                             NULL, NULL, NULL, NULL, NULL);
	        ErrFatalDisplayIf(err != 0, "DmDatabaseInfo");
		StrNCopy(items[type][nitems[type]].name, name, 39);
	    }
	}
    }

    DmCloseDatabase(ref);
}

static void DrawHelpButton(FormPtr frm, UInt id)
{
    RectangleType r;
    FontID font;
    Char ch;
    
    FrmGetObjectBounds(frm, FrmGetObjectIndex(frm, id), &r);
    WinEraseRectangle(&r, 4);
    font = FntSetFont(symbolFont);
    ch = symbolHelp;
    WinDrawChars(&ch, 1, r.topLeft.x + 3, 2);
    FntSetFont(font);
}

static void MainFormInit(FormPtr frm)
{
    ListPtr listP;

    if (nitems[ptype]) {
	listP = FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, liPrefs));
	
	LstSetListChoices(listP, NULL, nitems[ptype]);
	
	LstSetHeight(listP, 10);
	
	LstSetDrawFunction(listP, DrawPrefList);
	
	if (currentItem[ptype] != -1 && currentItem[ptype] < nitems[ptype])
	    LstSetSelection(listP, currentItem[ptype]);
	if (currentTopItem[ptype] != -1 && currentTopItem[ptype] < nitems[ptype])
	    LstSetTopItem(listP, currentTopItem[ptype]);
    }
    
}

static void RemoveEntryFromList(Word index)
{
    ListPtr listP = GetObjectPtr(liPrefs);
	
    MemPtrFree(items[ptype][index].name);
    if (index < nitems[ptype]) {
	MemMove(&items[ptype][index], &items[ptype][index+1], sizeof(items[ptype][index]) * (nitems[ptype] - index));
    }
    items[ptype][nitems[ptype]--].name = NULL;

    if (currentItem[ptype] == index)
	currentItem[ptype]--;

    LstSetListChoices(listP, NULL, nitems[ptype]);
    LstDrawList(listP);
}

static void DeleteRecord(Word index)
{
    UInt	cardNo;
    LocalID	dbID;
    Err		err;
    DmOpenRef	ref;
    DmSearchStateType	state;
    Boolean	newSearch = 1;
    Char	tmpbuf[50];
    
    switch (ptype) {
    case SAVED:
	err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'sprf', 'psys', false, &cardNo, &dbID);
	ErrFatalDisplayIf(err == dmErrCantFind, "No Saved Preferences");
	break;
    case UNSAVED:
	err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'pref', 'psys', false, &cardNo, &dbID);
	ErrFatalDisplayIf(err == dmErrCantFind, "No Unsaved Preferences");
	break;
    }


    ref = DmOpenDatabase(cardNo, dbID, dmModeReadWrite);
    
    if (ref == 0) {
	err = DmGetLastErr();
	StrCopy(tmpbuf, "DmOpenDatabase: condition ");
	StrIToA((tmpbuf + StrLen(tmpbuf)), err);
	ErrFatalDisplay(tmpbuf);
    }

    err = DmRemoveResource(ref, index);
    
    ErrNonFatalDisplayIf(err != 0, "DmRemoveResource");
    
    DmCloseDatabase(ref);
    
    RemoveEntryFromList(index);
}

static void SetNumber(FormPtr frm)
{
    char tmpbuf[20];
    RectangleType r;

    FrmGetObjectBounds(frm, FrmGetObjectIndex(frm, laNumber), &r);
    WinEraseRectangle(&r, 0);

    if (nitems[ptype] > 0) {
	if (currentItem[ptype] > 0)
	    StrIToA(tmpbuf, currentItem[ptype]+1);
	else
	    StrIToA(tmpbuf, 1);
	StrCat(tmpbuf, "/");
	StrIToA((tmpbuf + StrLen(tmpbuf)), nitems[ptype]);

	FrmCopyLabel(frm, laNumber, tmpbuf);
    }
}

static Boolean CreateMemo(const Char *text)
{
    Boolean retval;
    DmOpenRef dbref;
    UInt len;
    
    retval = false;
    len = StrLen(text) + 1;
    
    dbref = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
    if (dbref) {
	VoidHand h;
	UInt index;
	
	h = DmNewRecord(dbref, &index, len);
	if (h) {
	    VoidPtr p;
	    
	    p = MemHandleLock(h);
	    if (p) {
		retval = true;
		
		if (DmSet(p, 0, len, 0))
		    retval = false;

		if (DmWrite(p, 0, (VoidPtr) text, len))
		    retval = false;
		
		if (MemPtrUnlock(p))
		    retval = false;
	    }
	    
	    if (DmReleaseRecord(dbref, index, true))
		retval = false;
	    
	    if (!retval)
		DmDeleteRecord(dbref, index);
	}
	
	if (DmCloseDatabase(dbref))
	    retval = false;
    }
    
    return retval;
}

static Boolean ExportList(void)
{
    Boolean retval;
    Char *str;
    int i;
    
    retval = false;
    
    str = MemPtrNew(MEMOMAX);
    
    if (str == NULL)
	return false;
	
    MemSet(str, MEMOMAX, 0);
	
    StrCopy(str, "List of ");
    StrCat(str, (ptype == SAVED) ? "Saved" : "Unsaved");
    StrCat(str, " Preferences\n\n");
    
    for (i = 0; ((i < nitems[ptype]) && (StrLen(str) < ( MEMOMAX - MEMOLINE ))); i++) {
	Char creator[5];
	Char num[5];
#ifdef DISPLAY_ID_TOO
	Char    id[6];

	StrIToA(id, (UInt) (items[ptype][i].id));
#endif
	MemMove(creator, &items[ptype][i].creator, sizeof(items[ptype][i].creator));

	creator[4] = '\0';
	
	StrIToA(num, i + 1);
	
	StrCat(str, num);
	StrCat(str, ". ");

	StrCat(str, creator);
	StrCat(str, "\t");
#ifdef DISPLAY_ID_TOO
	StrCat(str, id);
	StrCat(str, "\t");
#endif
	StrCat(str, items[ptype][i].name);
	StrCat(str, "\n");
    }

    retval = CreateMemo(str);
    
    MemPtrFree(str);
    return (retval);
}

char hex[16] = {'0', '1' ,'2', '3', '4', '5', '6', '7', '8', '9', 'a',
                'b','c','d','e','f'};

static DmOpenRef openRecord(void);
static void closeRecord(DmOpenRef ref);

static void ExportRecordContent(Char *str)
{
    static DmOpenRef ref = NULL;

    char bytes[2];
    char line[LINEWIDTH + 1];
    unsigned char byte;
    int i;
    int pos;
    

    currentItem[ptype] = LstGetSelection(GetObjectPtr(liPrefs));
    currentTopItem[ptype] = LstGetTopItem(GetObjectPtr(liPrefs));

    ref = openRecord();

    i = 0;
    
    MemPtrSize(ptr);
    
    MemSet(line, LINEWIDTH, 0);
    
    for (pos = 0; pos < MemPtrSize(ptr); pos++) {
	byte     = ptr[pos];
	bytes[0] = hex[(byte >> 4) & 0xf];
	bytes[1] = hex[byte        & 0xf];
	
	StrCat(str, bytes);
	
	if ((byte > 31) && (byte < 127))
	    line[i] = byte;
	else
	    line[i] = '.';
	
	i++;
	
	if (i == LINEWIDTH) {
	    StrCat(str, "\t");
	    StrCat(str, line);
	    StrCat(str, "\n");
	    MemSet(line, LINEWIDTH, 0);
	    i = 0;
	} else
	    StrCat(str, " ");
    }
    StrCat(str, "\t");
    StrCat(str, line);
    StrCat(str, "\n");

    closeRecord(ref);
}

static Boolean ExportItem(void)
{
    Boolean retval;
    Char *str;
    Char num[5];
    Char creator[5];
    int idx;

#ifdef DISPLAY_ID_TOO
    Char    id[6];
#endif
    
    retval = false;
    
    str = MemPtrNew(MEMOMAX);
    
    if (str == NULL)
	return false;
	
    MemSet(str, MEMOMAX, 0);
	
    idx = currentItem[ptype];
    
    if (! (idx > 0))
	idx = 0;

    StrIToA(num, idx + 1);

    StrCopy(str, (ptype == SAVED) ? "Saved" : "Unsaved");
    StrCat(str, " Preferences Record # ");
    StrCat(str, num);
    StrCat(str, "\n\n");
    
#ifdef DISPLAY_ID_TOO
    StrIToA(id, (UInt) (items[ptype][idx].id));
#endif
    MemMove(creator, &items[ptype][idx].creator, sizeof(items[ptype][idx].creator));

    creator[4] = '\0';
	
    StrCat(str, creator);
    StrCat(str, "\t");
#ifdef DISPLAY_ID_TOO
    StrCat(str, id);
    StrCat(str, "\t");
#endif
    StrCat(str, items[ptype][idx].name);
    StrCat(str, "\n");

    ExportRecordContent(str);

    retval = CreateMemo(str);
    
    MemPtrFree(str);
    return (retval);
}

static Boolean MainFormHandleEvent (EventPtr e)
{
    Boolean handled = false;
    FormPtr frm = FrmGetActiveForm();
    
    CALLBACK_PROLOGUE

    switch (e->eType) {
    case frmOpenEvent:

	MainFormInit(frm);

	switch (ptype) {
	    case SAVED:
		CtlSetValue(FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, pbSaved)), true);
		break;
	    case UNSAVED:
		CtlSetValue(FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, pbUnsaved)), true);
		break;
	}


	FrmDrawForm(frm);

	SetNumber(frm);
	
	DrawHelpButton(frm, btHelp);
	
	handled = true;
	break;

    case menuEvent:
	MenuEraseStatus(NULL);

	switch(e->data.menu.itemID) {
	case mnAbout:
	    FrmAlert(alAbout);
	    break;
	case mnHelp:
	    FrmHelp(MainHelp);
	    break;
	case mnExportL:
	    ExportList();
	    break;
	case mnExportI:
	    ExportItem();
	    break;
	}

    	handled = true;
	break;

    case ctlSelectEvent:
	switch(e->data.ctlSelect.controlID) {
	case btDelete: {
	
	    Word sel;
	    Char creator[5];
	    CharPtr name = NULL;
	
	    if (nitems[ptype] > 0) {
		sel = LstGetSelection(GetObjectPtr(liPrefs));
		
		name = items[ptype][sel].name;
		MemMove(creator, &items[ptype][sel].creator, sizeof(items[ptype][sel].creator));
		
		if (!FrmCustomAlert(alVerify, name, creator, " ")) {
		    DeleteRecord(sel);
		    SetNumber(frm);
		}
	    }
	    handled = true;
	    }
	    break;
	case btView:
	    if (nitems[ptype] > 0) {

		currentItem[ptype] = LstGetSelection(GetObjectPtr(liPrefs));
		currentTopItem[ptype] = LstGetTopItem(GetObjectPtr(liPrefs));
		
		FrmGotoForm(ViewForm);
	    }
	    handled = true;
	    break;
	case pbSaved:
	    if (ptype == SAVED) {
		handled = true;
		break;
	    }
	    /* else fall through */
	case pbUnsaved: {
	    ListPtr listP = GetObjectPtr(liPrefs);
	    
	    if (e->data.ctlSelect.controlID == pbUnsaved && ptype == UNSAVED) {
		handled = true;
		break;
	    }
	    
	    currentItem[ptype]    = LstGetSelection(listP);
	    currentTopItem[ptype] = LstGetTopItem(listP);
	    if (ptype == UNSAVED)
		ptype = SAVED;
	    else
		ptype = UNSAVED;
	    LstSetListChoices(listP, NULL, nitems[ptype]);
	    if (currentItem[ptype] != -1 && currentItem[ptype] < nitems[ptype])
		LstSetSelection(listP, currentItem[ptype]);
	    else
		if (nitems[ptype] > 0)
		    LstSetSelection(listP, 0);
	    if (currentTopItem[ptype] != -1 && currentTopItem[ptype] < nitems[ptype])
		LstSetTopItem(listP, currentTopItem[ptype]);
	    else
		if (nitems[ptype] > 0)
		    LstSetTopItem(listP, 0);
	    LstDrawList(listP);
	    SetNumber(frm);
	    handled = true;
	    }
	    break;
	case btHelp:
	    FrmHelp(MainHelp);
	    break;
	}
	break;

    case lstSelectEvent:
	
	currentItem[ptype] = e->data.lstSelect.selection;
	SetNumber(frm);
	/* DON'T set handled */
	break;

    case keyDownEvent:
	switch (e->data.keyDown.chr) {

	case pageUpChr:
	case pageDownChr: {
	    ListPtr listP = GetObjectPtr(liPrefs);
	    
	    LstScrollList(listP, (e->data.keyDown.chr == pageUpChr) ? up : down,  LstGetVisibleItems(listP) - 1);
	    handled = true;
	    break;
	}
	}

    default:
        break;
    }

    CALLBACK_EPILOGUE

    return handled;
}

static void closeRecord(DmOpenRef ref)
{
    if (hand != NULL) {
	MemHandleUnlock(hand);
	hand =NULL;
    }
    if (0 && ptr != NULL) {
	MemPtrUnlock(ptr);
	ptr = NULL;
    }
    
    DmCloseDatabase(ref);
}


RectangleType dataRect = {{0, 20}, {160, 110}};

static void displayData()
{
    char	bytes[2];
    char	line[LINEWIDTH];
    char	type[4];
    unsigned char	byte;
    int		i,j;
    int		pos;
    
    i = 0;
    j = 0;
    
    
    WinEraseRectangle(&dataRect, 0);
    
    line[0] = hex[(offset >> 12) & 0xf];
    line[1] = hex[(offset >>  8) & 0xf];
    line[2] = hex[(offset >>  4) & 0xf];
    line[3] = hex[(offset >>  0) & 0xf];
    line[4] = ':';
    
    *((ULong*)type) = items[ptype][currentItem[ptype]].creator;
    
    WinDrawChars(type, 4, 30, 15);
    
    WinDrawChars(line, 5,  0, 15);
    
    MemPtrSize(ptr);
    
    MemSet(line, LINEWIDTH, ' ');
    
    for (pos=offset; pos < MemPtrSize(ptr); pos++) {
	byte     = ptr[pos];
	bytes[0] = hex[(byte >> 4) & 0xf];
	bytes[1] = hex[byte        & 0xf];
	
	WinDrawChars(bytes, 2, i * 12, 25 + j * 10);
	
	line[i] = byte;
	
	i++;
	
	if (i==LINEWIDTH) {
	    WinDrawChars(line, LINEWIDTH, 100, 25 + j * 10);
	    MemSet(line, LINEWIDTH, ' ');
	    i = 0;
	    j++;
	    if (j==LINEHEIGHT) {
		break;
	    }
	}
    }
    WinDrawChars(line, LINEWIDTH, 100, 25 + j * 10);
}

static DmOpenRef openRecord(void)
{
    Err		err;
    LocalID	chunkID;
    UInt	cardNo;
    LocalID	dbID;
    DmOpenRef	ref;
    Boolean	newSearch = 1;
    DmSearchStateType	state;
    char	tmpbuf[50];
    
    switch (ptype) {
    case SAVED:
	err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'sprf', 'psys', false, &cardNo, &dbID);
	ErrFatalDisplayIf(err == dmErrCantFind, "No Saved Preferences");
	break;
    case UNSAVED:
	err = DmGetNextDatabaseByTypeCreator(newSearch, &state, 'pref', 'psys', false, &cardNo, &dbID);
	ErrFatalDisplayIf(err == dmErrCantFind, "No Unsaved Preferences");
	break;
    }

    ref = DmOpenDatabase(cardNo, dbID, dmModeReadOnly);
    
    if (ref == 0) {
	err = DmGetLastErr();
	StrCopy(tmpbuf, "DmOpenDatabase: condition ");
	StrIToA((tmpbuf + StrLen(tmpbuf)), err);
	ErrFatalDisplay(tmpbuf);
    }
    
    err = DmResourceInfo(ref, currentItem[ptype], NULL, NULL, &chunkID);
    ErrFatalDisplayIf(err, "DmResourceInfo");
    
    if (MemLocalIDKind(chunkID) == memIDHandle) {
	hand = MemLocalIDToGlobal(chunkID, cardNo);
	ptr = MemHandleLock(hand);
    } else {
	ptr = MemLocalIDToGlobal(chunkID, cardNo);
    }
    
    return ref;
}


static void scrollButtons()
{
    if (offset + LINEWIDTH * LINEHEIGHT < MemPtrSize(ptr))
	CtlSetLabel(GetObjectPtr(btDown), "\002");
    else
	CtlSetLabel(GetObjectPtr(btDown), "\004");

    if (offset >= LINEWIDTH * LINEHEIGHT)
	CtlSetLabel(GetObjectPtr(btUp), "\001");
    else
	CtlSetLabel(GetObjectPtr(btUp), "\003");
}

static void scrollUp()
{
    if (offset >= LINEWIDTH * LINEHEIGHT)
        offset -= LINEWIDTH * LINEHEIGHT;
    scrollButtons();
    displayData();
}

static void scrollDown()
{
    if (offset + LINEWIDTH * LINEHEIGHT < MemPtrSize(ptr))
        offset += LINEWIDTH * LINEHEIGHT;
    scrollButtons();
    displayData();
}

static Boolean ViewFormHandleEvent (EventPtr e)
{
    Boolean handled = false;
    FormPtr frm;
    static DmOpenRef ref = NULL;
    
    CALLBACK_PROLOGUE

    switch (e->eType) {
    case frmOpenEvent:
	frm = FrmGetActiveForm();

	FrmDrawForm(frm);
	
	ref = openRecord();
	
	offset = 0;
	displayData();
	
	scrollButtons();
	
	handled = true;
	break;

    case menuEvent:
	MenuEraseStatus(NULL);

	switch(e->data.menu.itemID) {
	}

    	handled = true;
	break;

    case ctlSelectEvent:
	switch(e->data.ctlSelect.controlID) {
	case btDone:
	    closeRecord(ref);
	    FrmGotoForm(MainForm);
	case btUp:
        scrollUp();
	    handled = true;
	    break;
	case btDown:
        scrollDown();
	    handled = true;
	    break;
	}
	break;

    case keyDownEvent:
	switch (e->data.keyDown.chr) {

	case pageUpChr:
        scrollUp();
        handled = true;
        break;
	case pageDownChr:
        scrollDown();
        handled = true;
	    break;
	}

    default:
        break;
    }

    CALLBACK_EPILOGUE

    return handled;
}

static Boolean ApplicationHandleEvent(EventPtr e)
{
    FormPtr frm;
    Word    formId;
    Boolean handled = false;

    if (e->eType == frmLoadEvent) {
	formId = e->data.frmLoad.formID;
	frm = FrmInitForm(formId);
	FrmSetActiveForm(frm);

	switch(formId) {
	case MainForm:
	    FrmSetEventHandler(frm, MainFormHandleEvent);
	    break;
	case ViewForm:
	    FrmSetEventHandler(frm, ViewFormHandleEvent);
	    break;
	}
	handled = true;
    }

    return handled;
}

/* Get preferences, open (or create) app database */
static Word StartApplication(void)
{
    RectangleType r;

    WinDrawChars("Please wait...", 14, 45, 76);
    
    r.topLeft.y = 70;
    r.topLeft.x = 30;
    r.extent.x = 100;
    r.extent.y= 20;
    
    WinDrawRectangleFrame(dialogFrame ,&r);
    
    InitPrefList();
    
    r.topLeft.y = 50;
    r.topLeft.x = 10;
    r.extent.x = 140;
    r.extent.y= 60;

    WinEraseRectangle(&r, 0);

    FrmGotoForm(MainForm);
    return 0;
}

/* Save preferences, close forms, close app database */
static void StopApplication(void)
{
    int i;

    FrmSaveAllForms();
    FrmCloseAllForms();
    
    for (i=0; i < nitems[SAVED]; i++) {
	MemPtrFree(items[SAVED][i].name);
    }

    for (i=0; i < nitems[UNSAVED]; i++) {
	MemPtrFree(items[UNSAVED][i].name);
    }
    
}

/* The main event loop */
static void EventLoop(void)
{
    Word err;
    EventType e;

    do {
	EvtGetEvent(&e, evtWaitForever);
	if (! SysHandleEvent (&e))
	    if (! MenuHandleEvent (NULL, &e, &err))
		if (! ApplicationHandleEvent (&e))
		    FrmDispatchEvent (&e);
    } while (e.eType != appStopEvent);
}

/* Main entry point; it is unlikely you will need to change this except to
   handle other launch command codes */
DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
    Word err;

    if (cmd == sysAppLaunchCmdNormalLaunch) {

	err = StartApplication();
	if (err) return err;

	EventLoop();
	StopApplication();

    } else {
	return sysErrParamErr;
    }

    return 0;
}
